// API Error message and handler
var express = require('express');
var http = require('http');
var errorTransformer = require('./error_transformer');
var debug = require('debug')('api');

var error = {};

var initializeErrorResponse = function(error, message, errorCode, response) {
	var msg;
	if (typeof message === 'object' && message.hasOwnProperty('message') && message.hasOwnProperty('code')) {
		msg = message;
	} else {
		msg = {"message":message};		
		if (errorCode) {
			msg.code = errorCode;
		} else {
			msg.code = "3"; // if no errorCode, set msg.code to "3" (kUnavailableCommand)
		}
	}
	error.messages = [msg];
	if (typeof response !== 'undefined') {
		error.response = response;
	} else {
		error.response = {};
	}
}

/**
 * @apiDefine BadRequestError
 *
 * @apiError 400-BadRequestError The server cannot or will not process the request due to something that is perceived to be a client error.
 *
 * @apiErrorExample 400 Bad Request:
 *     HTTP/1.1 400 Bad Request
 *     {
 *       "error": "error detail"
 *     }
 */
 error.BadRequest = {
	message: "Bad Request",
	code: 400,
	handle: function(req, res, message, errorCode){
		var error = {};
		initializeErrorResponse(error, message ? message : this.message, errorCode);
		res.status(this.code).json(error);
	}
}


/**
 * @apiDefine UnauthorizedError
 *
 * @apiError 401-UnauthorizedError Unauthorized access to the api.
 *
 * @apiErrorExample 401 Unauthorized:
 *     HTTP/1.1 401 Unauthorized
 *     {
 *       "error": "Unauthorized"
 *     }
 */
 error.Unauthorized = {
	message: "Unauthorized",
	code: 401,
	handle: function(req, res, message, errorCode) {
		var error = {};
		initializeErrorResponse(error, message ? message : this.message, errorCode);
		res.status(this.code).json(error);
  	}
}
 /**
 * @apiDefine ForbiddenError
 *
 * @apiError 403-ForbiddenError Forbidden access to this resource.
 *
 * @apiErrorExample 403 Forbidden:
 *     HTTP/1.1 403 Forbidden
 *     {
 *       "error": "Forbidden"
 *     }
 */
  error.Forbidden = {
	message: "Forbidden",
	code: 403,
	handle: function(req, res, message){
		var error = {};
		initializeErrorResponse(error, message ? message : this.message);			
		res.status(this.code).json(error)
	}
}

 /**
 * @apiDefine NotFoundError
 *
 * @apiError 404-NotFoundError Object Not Found.
 *
 * @apiErrorExample 404 Not Found:
 *     HTTP/1.1 404 Not Found
 *     {
 *       "error": "NotFound"
 *     }
 */
 error.NotFound = {
	message: "Not Found",
	code: 404,
	handle: function(req, res, message, errorCode){
		var error = {};
		if (errorCode) {
			initializeErrorResponse(error, message ? message : this.message, errorCode);			
		} else {
			initializeErrorResponse(error, message ? message : this.message);			
		}
		res.status(this.code).json(error)
	}
}

/**
* 405 Method Not Allowed
* @apiDefine MethodNotAllowed
*
* @apiError 405-MethodNotAllowed Method Not Allowed.
*
* @apiErrorExample 405 Error-Response:
*     HTTP/1.1 405 Method Not Allowed
*	   Allow: GET,POST
*     {
*       "error": "Method Not Allowed"
*     }
*/

error.MethodNotAllowed = {
  message: "Method Not Allowed",
  code: 405,
  handle: function(req, res, allowedMethods){
	var error = {};
	initializeErrorResponse(error, allowedMethods ? "Allowed HTTP Methods: "+allowedMethods : this.message);
	res.status(this.code).set('Allow', allowedMethods).json(error);
  }
}

/**
* @apiDefine UnsupportedMediaType
*
* @apiError 415-UnsupportedMediaType Unsupported Media Type.
*
* @apiErrorExample 415 Unsupported Media Type:
*     HTTP/1.1 415 Unsupported Media Type
*     {
*       "error": "Unsupported Media Type"
*     }
*/
error.UnsupportedMediaType = {
 message: "Unsupported Media Type",
 code: 415,
 handle: function(req, res, message){
	var error = {};
	initializeErrorResponse(error, message ? message : this.message);
	res.status(this.code).json(error)
 }
}

/**
* @apiDefine FMServiceError
*
* @apiError 477-FMServiceError FileMaker operation exception as listed at http://www.filemaker.com/help/14/fmp/en/html/error_codes.html
*
* @apiErrorExample 477 FM Service Error:
*     HTTP/1.1 477 FM Service Error
*     {
*       "error": "Database Service Error",
*       "errorCode": "errorcode",
*     }
*/
error.FMServiceError = {
 message: "FileMaker Data API Engine Error",
 code: 500,
 handle: function(req, res, message, errorCode, response){
	 res.statusMessage = this.message;
	 var error = {};
	 initializeErrorResponse(error, message ? message : this.message, errorCode, response);
	 res.status(this.code).json(error)
 }
}

 /**
 * @apiDefine InternalServerError
 * @apiError  500-InternalServerError Internal Server Error
 * @apiErrorExample 500 Error-Response:
 *     HTTP/1.1 500 Internal Server Error
 *     {
 *       "error": "The server encountered an unexpected condition which prevented it from fulfilling the request."
 *     }
 */
 error.InternalServerError = {
 	message: 'Internal Server Error',
 	code: 500,
 	handle: function(req, res, message){
		var error = {};
		initializeErrorResponse(error, message ? message : this.message);
 		res.status(this.code).json(error)
 	}
 }

  /**
 * @apiDefine NotImplementedError
 * @apiError  501-NotImplementedError
 * @apiErrorExample 501 Error-Response:
 *     HTTP/1.1 501  Not Implemented Error
 *     {
 *       "error": "Not Implemented."
 *     }
 */


 var handleError = function(routine, req, res, message, errorCode, response){
 	message = message ? message : '';
 	if(error[routine] ){
 		error[routine].handle(req, res, message, errorCode, response);
 	} else {
 		error.InternalServerError.handle(req, res, message, errorCode);
 	}
 }


module.exports.handleError = handleError;
